/*
	Zarizeni:	FM-cache
	Procesor:	ATMega8
	Freq:		1 MHz (interni oscilator)
	Verze:		0.1


	Obsazene piny:
	-------------

		PB5(SPI->SCK)		\
		PB4(SPI->MISO)		|___	ISD1760PY
		PB3(SPI->MOSI)		|
		PB2(SPI->SS)		/

		PD5(DS1302->CE)		\
		PD6(DS1302->I/O)	|---	DS1302
		PD7(DS1302->SCLK)	/

		PB6, PB7			krystal 32 768 Hz pro citac 1

		PB1(FM-SWITCH)		zapinani FM vysilace
		PD0(LED-RED)		on/off -> nastaveni hodin/minut (minuty s krokem 5)
		PD1(LED-GREEN)		aktualni hodnota hodin/minut
		PD2(BUTTON1)		aktivace nastaveni dne a casu
		PD3(BUTTON2)		zmena hodnot dne a casu


	Funkce:
	------
		
		Hlavni funkce: kazdych 5 minut kontroluje cas a pokud je hodina
					   "00" a minuta mensi nez "20" prehraje zpravu.
		.
		Po pripojeni napajeni program skenuje tlacitka a ceka na jednu z akci:
		1) stisk BUTTON1 prepina mezi volbou nastaveni datumu a casu (postupne
		   YYYYMMDDhhmm). 
		2) stisk BUTTON2 pricita 1 k prave nastavovane hodnote.
		3) pri dlouhem stisku BUTTON2 se rozsviti  cervena dioda a spusti se nahravani
		   do audiopameti. Nahravani se ukonci pri zaplneni pameti nebo stiskem BUTTON2.
		4) pri soucasnem stisku BUTTON1 a BUTTON2 se prehraje ulozena sprava a zarizeni
		   se prepne do spanku viz "Hlavni funkce".
*/


#include "global.h"		//nacteni vsech potrebnych promennych a souboru s funkcemi

ISR(TIMER2_OVF_vect) {	//po docitani timeru se provede toto preruseni
	sleep_timer_stop();										//zastavim casovac, moje obsluha totiz muze trvat dlouho	
	timer_run_cnt = timer_run_cnt + 1;						//pro zjisteni jestli uz ubehlo 5 minut
	
	// kdyz jsou obe tlacitka stisknuta, zrusim sleep rezim (nespustim znovu citac a tak program neskoci do sleep rezimu)
	if ( !get_button_state(BUTTON1_D) & !get_button_state(BUTTON2_D) ) {

		set_led_state(LED_GREEN_D,1);	// bliknu ledkama
		set_led_state(LED_RED_D,1);
		_delay_ms(2000);				// cas at muzu pustit tlacitka
		set_led_state(LED_GREEN_D,0);
		set_led_state(LED_RED_D,0);

		timer_run_cnt = 0;				// vynuluju promenne
		cache_run_cnt = 0;
		return;							// opoustim obsluhu preruseni
	}	

	// kontrola jednou za 5 minut -> (5 * 60) / 8 (doba citani casovace) = 37.5
	if (timer_run_cnt == 38) {
		
		hour = ds1302_read_hour();							//prectu hodiny
		minute = ds1302_read_minute();						//prectu minuty
	
		if ( (hour == run_hour) & (minute < 0x14) ) {	//pokud je cas ve vybranem intervalu <hod:00,hod:20>
			
			if (cache_run_cnt < 3) {						//jeste jem zpravu neprehraval 3x
				isd_power_up();								//proberu audio obvod
				fm_transmitter(1);							//zapnu fm vysilac

					while (!isd_is_ready()) { _delay_ms(1000); }//cekam dokud neni audiopamet pripravena
					isd_play(0);								//prehravam
					while (isd_in_play()) { _delay_ms(1000); }	//cekam dokud se zprava neprehraje

				fm_transmitter(0);							//vypnu fm vysilac
				isd_power_down();							//uspim audio obvod

				cache_run_cnt = cache_run_cnt + 1;			//prictu pocitadlo spusteni kesky
			}
		} else {
					cache_run_cnt = 0;						//kdyz nejsem ve "spostecim" case, nuluju pocitadlo prehranych zprav
			   }
		
		timer_run_cnt = 0;									//nuluju pocitadlo 5 minut
	}

	sleep_timer_start();									//spustim znovu casovac
}

void main_init(void) {
	DDRD = DDRD | (1<<LED_RED_D) | (1<<LED_GREEN_D);	//vystupy
	DDRD = DDRD & ~(1<<BUTTON1_D);						//vstup
	DDRD = DDRD & ~(1<<BUTTON2_D);						//vstup
	//vsechno na 0
	PORTD = PORTD & ~(1<<BUTTON1_D);
	PORTD = PORTD & ~(1<<BUTTON2_D);
	PORTD = PORTD & ~(1<<LED_RED_D);
	PORTD = PORTD & ~(1<<LED_GREEN_D);
}

void write_TCNT2(unsigned char start) {		//zapise novou hodnotu a ceka nez se zapis provede

	TCNT2 = start;							//zapisu novy start
	while ((ASSR & 0x04) != 0x00) { ; }		//cekam, nez se zapis provede
};

void write_OCR2(unsigned char start) {

	OCR2 = start;
	while ((ASSR & 0x02) != 0x00) { ; }
};


void sleep_timer_init(void) {							//natavim si casovac, ktery me bude budit ze spanku
	//pouziju casovac 2 (8 bitu) se zdrojem signalu z externiho hodinoveho krystalu 32678 Hz

	ASSR = ASSR | (1<<AS2);		// aktivace asynchronniho citace (hodiny bude brat z krystalu 32678 Hz)
	write_TCNT2(0x00);			// nuluju registry
	write_OCR2(0x00);			// nuluju registry

	TCCR2 = 0x00;							// citac je vypnuty
	while ((ASSR & 0x01) != 0x00) { ; }		// cekam nez se zmena provede

	TIMSK = (1<<TOIE2);			// povolim jen preruseni pri preteceni citace

	ACSR = (1<<ACD);			// vypne komparator (sporim elektriku)
	//BODEN (hlidani urovne napeti) je vypnuto defaultne
	//WATCHDOG je take vypnuty defaultne
}


void sleep_timer_start(void) {				// spusti casovac
	_delay_us(100);							// pred novym uspanim musim alespon 1 TOSC1 (1/32768 = 30.5 us) 
											// cyklus pockat nez muzu znovu spustit sleep rezim -> radeji 100 us
	// N = 1024, 8bit -> 256
	// T = (1/fosc2) * 1024 * 256 = (1/32768)*1024*256 = 8 s
	write_TCNT2(0x00);							// nastavim pocatek citani	
	TCCR2 = (1<<CS22) | (1<<CS21) | (1<<CS20);	// spustim citac
	while ((ASSR & 0x01) != 0x00) { ; }			// cekam nez se zmena provede
}

void sleep_timer_stop(void) {				//zastavi citac

	TCCR2 = 0x00;							// vypny citac
	while ((ASSR & 0x01) != 0x00) { ; }		// cekam nez se zmena provede	
}

void sleep_mcu(void) {
	// SLEEP_MODE_STANDBY, SLEEP_MODE_IDLE, SLEEP_MODE_PWR_DOWN, SLEEP_MODE_PWR_SAVE		
	set_sleep_mode(SLEEP_MODE_PWR_SAVE);				//zvoli sleep rezim
    sleep_enable();										//uspi procesor
    sei();												//povoli preruseni
    sleep_cpu();										//procesor se uspi
    sleep_disable();									//po probuzeni se uspani zrusi
}

void eeprom_write (unsigned int address, unsigned char data) {
	while (EECR & (1<<EEWE)) { ; }						//cekam na dokonceni predchoziho zapisu
	EEAR = address;		
	EEDR = data;
	EECR = EECR | (1<<EEMWE);							//priprava zapisu do eeprom
	EECR = EECR | (1<<EEWE);							//start zapisu do eeprom
}

unsigned char eeprom_read (unsigned int address) {
	while (EECR & (1<<EEWE)) { ; }						//cekam na dokonceni predchoziho zapisu
	EEAR = address;
	EECR = EECR | (1<<EERE);							//start cteni z eeprom
	return EEDR;
}

unsigned char get_button_state(unsigned char port_pin) {
	if (PIND & (1<<port_pin)) {
	  return 1;
	} else {
			 return 0;
		   }
}

void set_led_state(unsigned char port_pin, unsigned char state) {
	if (state == 0) {
	  PORTD = PORTD & ~(1<<port_pin);
	} else {
			 PORTD = PORTD | (1<<port_pin);
	       }
}

void flash_value(unsigned char led, unsigned char value) {
	while (value > 0) {				//vyblikam hodnotu
		set_led_state(led,1);
		_delay_ms(300);
		set_led_state(led,0);
		_delay_ms(300);
		value = value - 1;
	}	
}

int main(void) {
	cli();							//zakazu globalni preruseni

	// inicializace vsech periferii
	main_init();					//inicializace tlacitek a ledek
	sleep_timer_init();				//inicializace casovace
	fm_switch_init();				//zapinani FM vysilace (defaultne na OFF)
	isd_init();						//isp sbernice pro komunikaci s ISD modulem
	ds1302_init();					//inicializace pinu pro komunikaci s obvodem realneho casu

	//inicializace promennych
	btn1_down_cnt = 0;				//pocitani doby stisku tlacitka 1
	btn2_down_cnt = 0;				//pocitani doby stisku tlacitka 2

	setting_index = 1;				//urcuje to co prave nastavuji (1=year,2=month,3=day,4=hour,5=minute)
	cache_run_cnt = 0;				//pocitadlo, koliktrat za den bylo prehrano hlaseni kesky
	timer_run_cnt = 0;				//pocitadlo, kolikrat docital casovac do konce (trva to 1 minutu)
	
	if (eeprom_read(EE_INIT) != EE_INIT_VAL) {		//kontrola, jestli uz jsou v eeprom ulozene nejake hodnoty
	  eeprom_write(EE_INIT,EE_INIT_VAL);			//ulozeni defaultnich hodnot
	  eeprom_write(EE_HOUR,0);			
	}
	run_hour		= eeprom_read(EE_HOUR);			//zjisteni hodnot z eeprom

	year 	= ds1302_read_year();
	month 	= ds1302_read_month();
	day		= ds1302_read_day();
	hour	= ds1302_read_hour();
	minute	= ds1302_read_minute();

	isd_power_down();				//uspim audio obvod

	while (1) {						//nekonecny cyklus
		
		if ((TCCR2 & 0x07) == 0x00) {		//keska jeste neni prepnuta do pracovniho modu (casovac nebezi)

						
			//-------------------------------------------------------------------------------------------------

			if (!get_button_state(BUTTON1_D)) { 	//zmacknute tlacitko pro zmenu nastavovane hodnoty
				
				if (btn1_down_cnt == 10) {			//kdyz tlacitko drzim dyl, ulozim nastavenou hodnotu do obvodu realneho casu

					flash_value(LED_GREEN_D,setting_index);		//vyblikam co se uklada
					switch(setting_index) {
						case 1:	//rok
								ds1302_set_date(year,month,day);
								year = ds1302_read_year();		//hodnotu prectu at overim, ze s spravne zapsala
								flash_value(LED_RED_D,year);	//vyblikat hodnotu roku
								break;
						case 2:	//mesic
								ds1302_set_date(year,month,day);
								month = ds1302_read_month();
								flash_value(LED_RED_D,month);
								break;
						case 3:	//den
								ds1302_set_date(year,month,day);
								day = ds1302_read_day();
								flash_value(LED_RED_D,day);
								break;
						case 4:	//hodina
								ds1302_set_time(hour,minute);
								hour = ds1302_read_hour();
								flash_value(LED_RED_D,hour);
								break;
						case 5:	//minuta
								ds1302_set_time(hour,minute);
								minute = ds1302_read_minute();
								flash_value(LED_RED_D,minute);
								break;
					} //switch
					
					btn1_down_cnt = btn1_down_cnt + 1;			//pridam 1 aby se tato sekce uz neopakovala
				} else {
						 if (btn1_down_cnt < 10) {				//kdyz uz se jednou ulozilo, podruhe ho neukladam
						  btn1_down_cnt = btn1_down_cnt + 1;	//kontrola, jak dlouho je tlacitko stisknute
						 }
					   }
			} else {	//tlacitko neni zmackle
						if ((btn1_down_cnt > 0) & (btn1_down_cnt < 10)) {		//tlacitko bylo zmacknuto jen kratce
							setting_index = setting_index + 1;					//zmena nastavovane hodnoty
							if (setting_index == 6) { setting_index = 1; }		//kontroluju "preteceni"
														
							flash_value(LED_GREEN_D,setting_index);				//vyblikat co se nastavuje
							switch(setting_index) {								//a k tomu hodnotu, jaka je v hodinach
								case 1:	//rok
										year = ds1302_read_year();
										flash_value(LED_RED_D,year);
										break;
								case 2:	//mesic
										month = ds1302_read_month();
										flash_value(LED_RED_D,month);
										break;
								case 3:	//den
										day = ds1302_read_day();
										flash_value(LED_RED_D,day);
										break;
								case 4:	//hodina
										hour = ds1302_read_hour();
										flash_value(LED_RED_D,hour);
										break;
								case 5:	//minuta
										minute = ds1302_read_minute();
										flash_value(LED_RED_D,minute);
										break;
							} //switch
						} //tlacitko stisknuto kratce
						btn1_down_cnt = 0;										//reset pocitadla
					}

			//-------------------------------------------------------------------------------------------------

			if (!get_button_state(BUTTON2_D)) {					//zmacknute tlacitko pro inkrementaci nastavovane hodnoty

				if (btn2_down_cnt == 10) {						//tlacitko uz drzim dlouho

					isd_power_up();								//proberu audio obvod
					while (!isd_is_ready()) { _delay_ms(1000); }//cekam dokud neni audiopamet pripravena

					isd_erase();								//vymazu celou pamet
					isd_record(0);								//nahravam
					set_led_state(LED_RED_D,1);					//rozsvicena ledka me informuje, ze nahravam
					_delay_ms(2000);							//cekam 2 vteriny at stihnu pustit cudlik nahravani
					while (isd_in_record()) {					//cekam dokud se zprava nenahraje
						_delay_ms(1000);
						if (!get_button_state(BUTTON2_D)) {		//pokud znovu zmacknu cudlik
							isd_stop();							//koncim nahravani
						}
					}
					set_led_state(LED_RED_D,0);					//zhasnu ledku

					//to co se mi nahralo si prehraju
					set_led_state(LED_GREEN_D,1);				//rozsvicena ledka me informuje, ze prehravam
					fm_transmitter(1);							//zapnu fm vysilac
					while (!isd_is_ready()) { _delay_ms(1000); }//cekam dokud neni audiopamet pripravena
					isd_play(0);								//prehravam s rozsvicenou LED
					while (isd_in_play()) { _delay_ms(1000); }	//cekam dokud se zprava neprehraje
					fm_transmitter(0);							//vypnu fm vysilac
					set_led_state(LED_GREEN_D,0);				//zhasnu ledku

					isd_power_down();							//uspim ISD obvod

					btn2_down_cnt = btn2_down_cnt + 1;			//pridam 1 aby se tato sekce uz neopakovala
				} else {
							if (btn2_down_cnt < 10) {
							  btn2_down_cnt = btn2_down_cnt + 1;	//pocitam, jak dlouho mam stiknuto
							}
						}
			} else {	//tlacitko neni stiknute
					  if ((btn2_down_cnt > 0) & (btn2_down_cnt < 10)) {	 
						switch(setting_index) {
							case 1:	//rok
									year = year + 1;
									if (year == 100) { year = 0; }		//rok je v rozmezi 0-99
									break;
							case 2:	//mesic
									month = month + 1;
									if (month == 13) { month = 1; }		//mesic je v rozmezi 1-12
									break;
							case 3:	//den
									day = day + 1;
									if (day == 32) { day = 1; }			//mesic je v rozmezi 1-31
									break;
							case 4:	//hodina
									hour = hour + 1;
									if (hour == 24) { hour = 0; }		//hodina je v rozmezi 0-23
									break;
							case 5:	//minuta
									minute = minute + 1;
									if (minute == 60) { minute = 0; }	//minuta je v rozmezi 0-59
									break;
	 				    } //switch
					  }
					  btn2_down_cnt = 0;	
				   }

			//-------------------------------------------------------------------------------------------------

			if ( !get_button_state(BUTTON1_D) & !get_button_state(BUTTON2_D) ) {		//mam stiknuta obe tlacitka, prepnu se do provozniho modu
				
				//nastavit interval nebo spustit kesku?
				set_led_state(LED_GREEN_D,1);
				_delay_ms(5000);														//cekam 5 vterin

				if ( !get_button_state(BUTTON1_D) & !get_button_state(BUTTON2_D) ) {	//kontroluju stav tlacitek

					//i po 5ti vterinach jsou tlacitka stisknuta -> chci zmenit cas spousteni
					set_led_state(LED_GREEN_D,0);										//zhasnu ledku -> info, ze budu nastavovat spousteci cas
					_delay_ms(1000);													//chvilku pockam
					flash_value(LED_RED_D,run_hour);									//vyblikam aktalne nastavenou hodnotu
					set_led_state(LED_GREEN_D,1);										//rozsvitim ledku -> info, ze cekam na navoleni nove hodnoty

					unsigned char btn2_last_state = 0;
					while (get_button_state(BUTTON1_D)) {								//dokud nezmacknu tlacitko 1, pocitam stisky tlacitka 2
						if (!get_button_state(BUTTON2_D)) {								//zmackl jsem tlacitko 2
							btn2_last_state = 1;										//zapamatuju si to
						} else {														//tlacitko neni stisknute
								  if (btn2_last_state == 1) {							//a predtim bylo
								  	 btn2_last_state = 0;								//nuluju stav tlacitka
									 run_hour = run_hour + 1;							//pricitam hodinu
									 if (run_hour == 24) { run_hour = 0; }				//kontroluju meze
								  }
							   }
						_delay_ms(100);													//tlactika kontroluju 10x za vterinu
					} //while
				
					//zmacknuto tlacitko 1 -> ukladam novou hodnotu spousteni kesky
					eeprom_write(EE_HOUR,run_hour);
					flash_value(LED_RED_D,run_hour);									//a novou hodnotu vyblikam
					set_led_state(LED_GREEN_D,0);										//zhasinam ledku

					btn1_down_cnt = 0;			//nuluju "delku" stisku tlacitka
					btn2_down_cnt = 0;

				} else {	//chci aktivovat kesku					
							//to co se mi nahralo si prehraju
							set_led_state(LED_RED_D,1);
							isd_power_up();								//proberu audio obvod
							fm_transmitter(1);							//zapnu fm vysilac
							while (!isd_is_ready()) { _delay_ms(1000); }//cekam dokud neni audiopamet pripravena
							isd_play(0);								//prehravam
							while (isd_in_play()) { _delay_ms(1000); }	//cekam dokud se zprava neprehraje
							set_led_state(LED_GREEN_D,0);
							set_led_state(LED_RED_D,0);
							fm_transmitter(0);							//vypnu fm vysilac
							isd_power_down();							//uspim adio obvod
							
							sleep_timer_start();		//spustim casovac
							sleep_mcu();				//uspim procesor

							btn1_down_cnt = 0;			//nuluju "delku" stisku tlacitka
							btn2_down_cnt = 0;
						}
			}

			//-------------------------------------------------------------------------------------------------

			_delay_ms(100);		//kazdy beh cekam cekam 0.1 s nez zacnu zjistovat stavy tlacitek

		} else {							//keska je prepnuta do pracovniho modu (casovac bezi)	
				sleep_mcu();				//uspim procesor				 
				btn1_down_cnt = 0;			//nuluju "delku" stisku tlacitka
				btn2_down_cnt = 0;
			   }

	} //konec od while
	return 0;
}
